<?php
/**
 * Daily Database Backup Script
 * 
 * Setup cron job (Linux):
 * 0 2 * * * php /path/to/sms/cron/backup.php
 * 
 * Windows Task Scheduler:
 * php C:\path\to\sms\cron\backup.php
 */

// Prevent web access
if (php_sapi_name() !== 'cli') {
    die("This script can only be run from command line");
}

require_once __DIR__ . '/../config/database.php';

$backup_dir = __DIR__ . '/../storage/backups/';

// Create backup directory if not exists
if (!is_dir($backup_dir)) {
    mkdir($backup_dir, 0755, true);
}

$filename = $backup_dir . 'sms_backup_' . date('Y-m-d_H-i-s') . '.sql';
$gzip_filename = $filename . '.gz';

echo "Starting backup...\n";
echo "Database: " . DB_NAME . "\n";
echo "Backup file: " . $filename . "\n";

// Create backup using mysqldump
$command = "mysqldump --user=" . escapeshellarg(DB_USER) . 
           " --password=" . escapeshellarg(DB_PASS) . 
           " --host=" . escapeshellarg(DB_HOST) . 
           " --single-transaction --quick --lock-tables=false " . 
           escapeshellarg(DB_NAME) . " > " . escapeshellarg($filename);

exec($command, $output, $return_var);

if ($return_var === 0) {
    // Compress backup
    if (function_exists('gzopen')) {
        $source = fopen($filename, 'rb');
        $dest = gzopen($gzip_filename, 'wb9');
        
        while (!feof($source)) {
            gzwrite($dest, fread($source, 4096));
        }
        
        fclose($source);
        gzclose($dest);
        unlink($filename); // Remove uncompressed file
        
        echo "✓ Backup successful (compressed): " . $gzip_filename . "\n";
        echo "  Size: " . round(filesize($gzip_filename) / 1024, 2) . " KB\n";
    } else {
        echo "✓ Backup successful: " . $filename . "\n";
        echo "  Size: " . round(filesize($filename) / 1024, 2) . " KB\n";
    }
    
    // Delete backups older than 30 days
    echo "\nCleaning old backups...\n";
    $files = glob($backup_dir . 'sms_backup_*');
    $deleted = 0;
    
    foreach ($files as $file) {
        if (filemtime($file) < strtotime('-30 days')) {
            unlink($file);
            echo "  Deleted: " . basename($file) . "\n";
            $deleted++;
        }
    }
    
    echo "  Total deleted: " . $deleted . " files\n";
    echo "\nBackup completed successfully!\n";
    
} else {
    echo "✗ Backup failed!\n";
    echo "Error code: " . $return_var . "\n";
    echo "Output: " . implode("\n", $output) . "\n";
    exit(1);
}
?>