<?php
/**
 * Transport Model
 * Handles all transport-related database operations
 */
require_once __DIR__ . '/../../core/Database.php';

class TransportModel {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    // Vehicles
    public function getAllVehicles() {
        $stmt = $this->db->query("SELECT * FROM vehicles ORDER BY vehicle_no ASC");
        return $stmt->fetchAll();
    }

    public function getVehicleById($id) {
        $stmt = $this->db->prepare("SELECT * FROM vehicles WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    public function createVehicle($data) {
        $stmt = $this->db->prepare("INSERT INTO vehicles (vehicle_no, vehicle_type, capacity, driver_name, driver_phone, route_name, status) VALUES (?, ?, ?, ?, ?, ?, ?)");
        return $stmt->execute([$data['vehicle_no'], $data['vehicle_type'], $data['capacity'], $data['driver_name'], $data['driver_phone'], $data['route_name'], $data['status']]);
    }

    public function updateVehicle($id, $data) {
        $stmt = $this->db->prepare("UPDATE vehicles SET vehicle_no=?, vehicle_type=?, capacity=?, driver_name=?, driver_phone=?, route_name=?, status=? WHERE id=?");
        return $stmt->execute([$data['vehicle_no'], $data['vehicle_type'], $data['capacity'], $data['driver_name'], $data['driver_phone'], $data['route_name'], $data['status'], $id]);
    }

    public function deleteVehicle($id) {
        $stmt = $this->db->prepare("DELETE FROM vehicles WHERE id = ?");
        return $stmt->execute([$id]);
    }

    // Routes
    public function getAllRoutes() {
        $stmt = $this->db->query("SELECT * FROM transport_routes ORDER BY route_name ASC");
        return $stmt->fetchAll();
    }

    public function getRouteById($id) {
        $stmt = $this->db->prepare("SELECT * FROM transport_routes WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    public function createRoute($data) {
        $stmt = $this->db->prepare("INSERT INTO transport_routes (route_name, start_point, end_point, distance_km, estimated_time, fare_amount, status) VALUES (?, ?, ?, ?, ?, ?, ?)");
        return $stmt->execute([$data['route_name'], $data['start_point'], $data['end_point'], $data['distance_km'], $data['estimated_time'], $data['fare_amount'], $data['status']]);
    }

    public function updateRoute($id, $data) {
        $stmt = $this->db->prepare("UPDATE transport_routes SET route_name=?, start_point=?, end_point=?, distance_km=?, estimated_time=?, fare_amount=?, status=? WHERE id=?");
        return $stmt->execute([$data['route_name'], $data['start_point'], $data['end_point'], $data['distance_km'], $data['estimated_time'], $data['fare_amount'], $data['status'], $id]);
    }

    public function deleteRoute($id) {
        $stmt = $this->db->prepare("DELETE FROM transport_routes WHERE id = ?");
        return $stmt->execute([$id]);
    }

    // Student Transport
    public function assignTransport($data) {
        $stmt = $this->db->prepare("INSERT INTO student_transport (student_id, vehicle_id, route_id, pickup_point, drop_point, monthly_fee, start_date, end_date, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
        return $stmt->execute([$data['student_id'], $data['vehicle_id'], $data['route_id'], $data['pickup_point'], $data['drop_point'], $data['monthly_fee'], $data['start_date'], $data['end_date'], $data['status']]);
    }

    public function getStudentTransport($student_id) {
        $stmt = $this->db->prepare("SELECT st.*, v.vehicle_no, v.vehicle_type, r.route_name 
                                    FROM student_transport st 
                                    JOIN vehicles v ON st.vehicle_id = v.id 
                                    JOIN transport_routes r ON st.route_id = r.id 
                                    WHERE st.student_id = ? AND st.status = 1");
        $stmt->execute([$student_id]);
        return $stmt->fetch();
    }

    public function getAllStudentTransport() {
        $stmt = $this->db->query("SELECT st.*, s.first_name, s.last_name, s.admission_no, v.vehicle_no, r.route_name 
                                  FROM student_transport st 
                                  JOIN students s ON st.student_id = s.id 
                                  JOIN vehicles v ON st.vehicle_id = v.id 
                                  JOIN transport_routes r ON st.route_id = r.id 
                                  WHERE st.status = 1 
                                  ORDER BY st.created_at DESC");
        return $stmt->fetchAll();
    }

    public function updateTransportStatus($id, $status) {
        $stmt = $this->db->prepare("UPDATE student_transport SET status = ? WHERE id = ?");
        return $stmt->execute([$status, $id]);
    }

    public function getVehicleCapacity($vehicle_id) {
        $stmt = $this->db->prepare("SELECT capacity FROM vehicles WHERE id = ?");
        $stmt->execute([$vehicle_id]);
        return $stmt->fetchColumn();
    }

    public function getVehicleOccupancy($vehicle_id) {
        $stmt = $this->db->prepare("SELECT COUNT(*) FROM student_transport WHERE vehicle_id = ? AND status = 1");
        $stmt->execute([$vehicle_id]);
        return $stmt->fetchColumn();
    }
}