<?php
/**
 * Library Model
 * Handles all library-related database operations
 */
require_once __DIR__ . '/../../core/Database.php';

class LibraryModel {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    // Books
    public function getAllBooks() {
        $stmt = $this->db->query("SELECT * FROM library_books ORDER BY book_title ASC");
        return $stmt->fetchAll();
    }

    public function getBookById($id) {
        $stmt = $this->db->prepare("SELECT * FROM library_books WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    public function searchBooks($keyword) {
        $stmt = $this->db->prepare("SELECT * FROM library_books WHERE book_title LIKE ? OR author LIKE ? OR isbn LIKE ?");
        $keyword = "%$keyword%";
        $stmt->execute([$keyword, $keyword, $keyword]);
        return $stmt->fetchAll();
    }

    public function createBook($data) {
        $stmt = $this->db->prepare("INSERT INTO library_books (book_title, author, isbn, publisher, publish_year, category, total_copies, available_copies, rack_number, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        return $stmt->execute([$data['book_title'], $data['author'], $data['isbn'], $data['publisher'], $data['publish_year'], $data['category'], $data['total_copies'], $data['total_copies'], $data['rack_number'], $data['status']]);
    }

    public function updateBook($id, $data) {
        $stmt = $this->db->prepare("UPDATE library_books SET book_title=?, author=?, isbn=?, publisher=?, publish_year=?, category=?, total_copies=?, available_copies=?, rack_number=?, status=? WHERE id=?");
        return $stmt->execute([$data['book_title'], $data['author'], $data['isbn'], $data['publisher'], $data['publish_year'], $data['category'], $data['total_copies'], $data['available_copies'], $data['rack_number'], $data['status'], $id]);
    }

    public function deleteBook($id) {
        $stmt = $this->db->prepare("DELETE FROM library_books WHERE id = ?");
        return $stmt->execute([$id]);
    }

    public function updateAvailability($id, $change) {
        $stmt = $this->db->prepare("UPDATE library_books SET available_copies = available_copies + ? WHERE id = ?");
        return $stmt->execute([$change, $id]);
    }

    // Book Issues
    public function issueBook($data) {
        $stmt = $this->db->prepare("INSERT INTO book_issues (book_id, student_id, issue_date, due_date, status, created_by) VALUES (?, ?, ?, ?, 'Issued', ?)");
        return $stmt->execute([$data['book_id'], $data['student_id'], $data['issue_date'], $data['due_date'], $data['created_by']]);
    }

    public function returnBook($id, $return_date, $fine) {
        $stmt = $this->db->prepare("UPDATE book_issues SET return_date=?, fine_amount=?, status='Returned' WHERE id=?");
        return $stmt->execute([$return_date, $fine, $id]);
    }

    public function getBookIssues($book_id = null, $student_id = null, $status = null) {
        $sql = "SELECT bi.*, lb.book_title, lb.isbn, s.first_name, s.last_name, s.admission_no, u.full_name as issued_by
                FROM book_issues bi 
                JOIN library_books lb ON bi.book_id = lb.id 
                JOIN students s ON bi.student_id = s.id 
                JOIN users u ON bi.created_by = u.id 
                WHERE 1=1";
        
        $params = [];
        if ($book_id) {
            $sql .= " AND bi.book_id = ?";
            $params[] = $book_id;
        }
        if ($student_id) {
            $sql .= " AND bi.student_id = ?";
            $params[] = $student_id;
        }
        if ($status) {
            $sql .= " AND bi.status = ?";
            $params[] = $status;
        }
        
        $sql .= " ORDER BY bi.issue_date DESC";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll();
    }

    public function getIssueById($id) {
        $stmt = $this->db->prepare("SELECT bi.*, lb.book_title, lb.isbn, s.first_name, s.last_name, s.admission_no 
                                    FROM book_issues bi 
                                    JOIN library_books lb ON bi.book_id = lb.id 
                                    JOIN students s ON bi.student_id = s.id 
                                    WHERE bi.id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    public function getOverdueBooks() {
        $stmt = $this->db->query("SELECT bi.*, lb.book_title, s.first_name, s.last_name, s.admission_no,
                                  DATEDIFF(CURDATE(), bi.due_date) as days_overdue
                                  FROM book_issues bi 
                                  JOIN library_books lb ON bi.book_id = lb.id 
                                  JOIN students s ON bi.student_id = s.id 
                                  WHERE bi.status = 'Issued' AND bi.due_date < CURDATE()
                                  ORDER BY bi.due_date ASC");
        return $stmt->fetchAll();
    }

    public function calculateFine($due_date, $return_date, $fine_per_day = 10) {
        $due = new DateTime($due_date);
        $return = new DateTime($return_date);
        $diff = $due->diff($return);
        $days = $diff->days;
        
        if ($days > 0) {
            return $days * $fine_per_day;
        }
        return 0;
    }

    public function getStudentIssues($student_id) {
        $stmt = $this->db->prepare("SELECT bi.*, lb.book_title, lb.isbn 
                                    FROM book_issues bi 
                                    JOIN library_books lb ON bi.book_id = lb.id 
                                    WHERE bi.student_id = ? AND bi.status = 'Issued'");
        $stmt->execute([$student_id]);
        return $stmt->fetchAll();
    }

    public function getStudentIssueCount($student_id) {
        $stmt = $this->db->prepare("SELECT COUNT(*) FROM book_issues WHERE student_id = ? AND status = 'Issued'");
        $stmt->execute([$student_id]);
        return $stmt->fetchColumn();
    }
}