<?php
/**
 * Boarding/Hostel Model
 * Handles all boarding-related database operations
 */
require_once __DIR__ . '/../../core/Database.php';

class BoardingModel {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    // Hostels
    public function getAllHostels() {
        $stmt = $this->db->query("SELECT h.*, 
                                  (SELECT COUNT(*) FROM rooms WHERE hostel_id = h.id) as total_rooms,
                                  (SELECT SUM(current_occupancy) FROM rooms WHERE hostel_id = h.id) as current_occupancy
                                  FROM hostels h ORDER BY h.name ASC");
        return $stmt->fetchAll();
    }

    public function getHostelById($id) {
        $stmt = $this->db->prepare("SELECT * FROM hostels WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    public function createHostel($data) {
        $stmt = $this->db->prepare("INSERT INTO hostels (name, type, total_rooms, total_capacity, warden_name, warden_phone, address, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
        return $stmt->execute([$data['name'], $data['type'], $data['total_rooms'], $data['total_capacity'], $data['warden_name'], $data['warden_phone'], $data['address'], $data['status']]);
    }

    public function updateHostel($id, $data) {
        $stmt = $this->db->prepare("UPDATE hostels SET name=?, type=?, total_rooms=?, total_capacity=?, warden_name=?, warden_phone=?, address=?, status=? WHERE id=?");
        return $stmt->execute([$data['name'], $data['type'], $data['total_rooms'], $data['total_capacity'], $data['warden_name'], $data['warden_phone'], $data['address'], $data['status'], $id]);
    }

    public function deleteHostel($id) {
        $stmt = $this->db->prepare("DELETE FROM hostels WHERE id = ?");
        return $stmt->execute([$id]);
    }

    // Rooms
    public function getAllRooms($hostel_id = null) {
        if ($hostel_id) {
            $stmt = $this->db->prepare("SELECT r.*, h.name as hostel_name FROM rooms r 
                                        JOIN hostels h ON r.hostel_id = h.id 
                                        WHERE r.hostel_id = ? ORDER BY r.room_number ASC");
            $stmt->execute([$hostel_id]);
        } else {
            $stmt = $this->db->query("SELECT r.*, h.name as hostel_name FROM rooms r 
                                      JOIN hostels h ON r.hostel_id = h.id 
                                      ORDER BY h.name, r.room_number ASC");
        }
        return $stmt->fetchAll();
    }

    public function getRoomById($id) {
        $stmt = $this->db->prepare("SELECT r.*, h.name as hostel_name FROM rooms r 
                                    JOIN hostels h ON r.hostel_id = h.id 
                                    WHERE r.id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    public function createRoom($data) {
        $stmt = $this->db->prepare("INSERT INTO rooms (hostel_id, room_number, floor, capacity, room_type, facilities, status) VALUES (?, ?, ?, ?, ?, ?, ?)");
        return $stmt->execute([$data['hostel_id'], $data['room_number'], $data['floor'], $data['capacity'], $data['room_type'], $data['facilities'], $data['status']]);
    }

    public function updateRoom($id, $data) {
        $stmt = $this->db->prepare("UPDATE rooms SET room_number=?, floor=?, capacity=?, room_type=?, facilities=?, status=? WHERE id=?");
        return $stmt->execute([$data['room_number'], $data['floor'], $data['capacity'], $data['room_type'], $data['facilities'], $data['status'], $id]);
    }

    public function updateOccupancy($id, $change) {
        $stmt = $this->db->prepare("UPDATE rooms SET current_occupancy = current_occupancy + ? WHERE id = ?");
        return $stmt->execute([$change, $id]);
    }

    public function deleteRoom($id) {
        $stmt = $this->db->prepare("DELETE FROM rooms WHERE id = ?");
        return $stmt->execute([$id]);
    }

    public function getAvailableRooms() {
        $stmt = $this->db->query("SELECT r.*, h.name as hostel_name, h.type as hostel_type 
                                  FROM rooms r 
                                  JOIN hostels h ON r.hostel_id = h.id 
                                  WHERE r.current_occupancy < r.capacity AND r.status = 1 
                                  ORDER BY h.name, r.room_number ASC");
        return $stmt->fetchAll();
    }

    // Boarding Students
    public function assignStudent($data) {
        $stmt = $this->db->prepare("INSERT INTO boarding_students (student_id, room_id, admission_date, boarding_type, monthly_fee, status, remarks, created_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
        return $stmt->execute([$data['student_id'], $data['room_id'], $data['admission_date'], $data['boarding_type'], $data['monthly_fee'], $data['status'] ?? 1, $data['remarks'] ?? null, $data['created_by']]);
    }

    public function getBoardingStudents() {
        $stmt = $this->db->query("SELECT bs.*, s.first_name, s.last_name, s.admission_no, r.room_number, h.name as hostel_name 
                                  FROM boarding_students bs 
                                  JOIN students s ON bs.student_id = s.id 
                                  JOIN rooms r ON bs.room_id = r.id 
                                  JOIN hostels h ON r.hostel_id = h.id 
                                  WHERE bs.status = 1 
                                  ORDER BY bs.admission_date DESC");
        return $stmt->fetchAll();
    }

    public function getStudentBoarding($student_id) {
        $stmt = $this->db->prepare("SELECT bs.*, r.room_number, h.name as hostel_name, h.type as hostel_type 
                                    FROM boarding_students bs 
                                    JOIN rooms r ON bs.room_id = r.id 
                                    JOIN hostels h ON r.hostel_id = h.id 
                                    WHERE bs.student_id = ? AND bs.status = 1");
        $stmt->execute([$student_id]);
        return $stmt->fetch();
    }

    public function checkoutStudent($id, $checkout_date) {
        $stmt = $this->db->prepare("UPDATE boarding_students SET checkout_date=?, status=0 WHERE id=?");
        return $stmt->execute([$checkout_date, $id]);
    }

    public function getBoardingById($id) {
        $stmt = $this->db->prepare("SELECT bs.*, s.first_name, s.last_name, s.admission_no, r.room_number, h.name as hostel_name 
                                    FROM boarding_students bs 
                                    JOIN students s ON bs.student_id = s.id 
                                    JOIN rooms r ON bs.room_id = r.id 
                                    JOIN hostels h ON r.hostel_id = h.id 
                                    WHERE bs.id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    // Room Inspections
    public function createInspection($data) {
        $stmt = $this->db->prepare("INSERT INTO room_inspections (room_id, inspector_id, inspection_date, cleanliness, maintenance_issues, action_taken, status) VALUES (?, ?, ?, ?, ?, ?, ?)");
        return $stmt->execute([$data['room_id'], $data['inspector_id'], $data['inspection_date'], $data['cleanliness'], $data['maintenance_issues'], $data['action_taken'], $data['status']]);
    }

    public function getRoomInspections($room_id) {
        $stmt = $this->db->prepare("SELECT ri.*, u.full_name as inspector_name FROM room_inspections ri 
                                    JOIN users u ON ri.inspector_id = u.id 
                                    WHERE ri.room_id = ? ORDER BY ri.inspection_date DESC");
        $stmt->execute([$room_id]);
        return $stmt->fetchAll();
    }

    public function updateInspectionStatus($id, $status) {
        $stmt = $this->db->prepare("UPDATE room_inspections SET status = ? WHERE id = ?");
        return $stmt->execute([$status, $id]);
    }

    // Statistics
    public function getTotalBoardingStudents() {
        $stmt = $this->db->query("SELECT COUNT(*) FROM boarding_students WHERE status = 1");
        return $stmt->fetchColumn();
    }

    public function getHostelOccupancy($hostel_id) {
        $stmt = $this->db->prepare("SELECT SUM(current_occupancy) as occupied, SUM(capacity) as capacity FROM rooms WHERE hostel_id = ?");
        $stmt->execute([$hostel_id]);
        return $stmt->fetch();
    }

    public function getAvailableBeds() {
        $stmt = $this->db->query("SELECT SUM(capacity - current_occupancy) FROM rooms WHERE status = 1");
        return $stmt->fetchColumn();
    }
}