<?php
/**
 * Attendance Model
 * Handles all attendance-related database operations
 */
require_once __DIR__ . '/../../core/Database.php';

class AttendanceModel {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    public function save($student_id, $date, $status, $remarks = null) {
        $stmt = $this->db->prepare("INSERT INTO student_attendance (student_id, date, status, remarks) 
                                    VALUES (?, ?, ?, ?) 
                                    ON DUPLICATE KEY UPDATE status=?, remarks=?");
        return $stmt->execute([$student_id, $date, $status, $remarks, $status, $remarks]);
    }

    public function bulkSave($attendance_data) {
        try {
            foreach ($attendance_data as $data) {
                $this->save($data['student_id'], $data['date'], $data['status'], $data['remarks']);
            }
            return true;
        } catch (Exception $e) {
            error_log($e->getMessage());
            return false;
        }
    }

    public function getByClassAndDate($class_id, $date) {
        $stmt = $this->db->prepare("SELECT sa.*, s.first_name, s.last_name, s.admission_no 
                                    FROM student_attendance sa 
                                    JOIN students s ON sa.student_id = s.id 
                                    WHERE s.class_id = ? AND sa.date = ?");
        $stmt->execute([$class_id, $date]);
        return $stmt->fetchAll();
    }

    public function getByStudent($student_id, $start_date = null, $end_date = null) {
        $sql = "SELECT * FROM student_attendance WHERE student_id = ?";
        $params = [$student_id];
        
        if ($start_date) {
            $sql .= " AND date >= ?";
            $params[] = $start_date;
        }
        if ($end_date) {
            $sql .= " AND date <= ?";
            $params[] = $end_date;
        }
        
        $sql .= " ORDER BY date DESC";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll();
    }

    public function getAttendanceRate($student_id, $month = null, $year = null) {
        $sql = "SELECT 
                COUNT(*) as total_days,
                SUM(CASE WHEN status = 'Present' THEN 1 ELSE 0 END) as present_days
                FROM student_attendance 
                WHERE student_id = ?";
        
        $params = [$student_id];
        
        if ($month && $year) {
            $sql .= " AND MONTH(date) = ? AND YEAR(date) = ?";
            $params[] = $month;
            $params[] = $year;
        }
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        $result = $stmt->fetch();
        
        if ($result && $result['total_days'] > 0) {
            return round(($result['present_days'] / $result['total_days']) * 100, 2);
        }
        return 0;
    }

    public function getClassAttendanceSummary($class_id, $date) {
        $stmt = $this->db->prepare("SELECT 
                                    COUNT(*) as total_students,
                                    SUM(CASE WHEN status = 'Present' THEN 1 ELSE 0 END) as present,
                                    SUM(CASE WHEN status = 'Absent' THEN 1 ELSE 0 END) as absent,
                                    SUM(CASE WHEN status = 'Late' THEN 1 ELSE 0 END) as late,
                                    SUM(CASE WHEN status = 'Excused' THEN 1 ELSE 0 END) as excused
                                    FROM student_attendance sa 
                                    JOIN students s ON sa.student_id = s.id 
                                    WHERE s.class_id = ? AND sa.date = ?");
        $stmt->execute([$class_id, $date]);
        return $stmt->fetch();
    }

    public function getMonthlyAttendance($student_id, $year) {
        $stmt = $this->db->prepare("SELECT 
                                    MONTH(date) as month,
                                    COUNT(*) as total_days,
                                    SUM(CASE WHEN status = 'Present' THEN 1 ELSE 0 END) as present_days
                                    FROM student_attendance 
                                    WHERE student_id = ? AND YEAR(date) = ?
                                    GROUP BY MONTH(date)
                                    ORDER BY MONTH(date)");
        $stmt->execute([$student_id, $year]);
        return $stmt->fetchAll();
    }
}