<?php
/**
 * Medical/Infirmary Controller
 * Handles all medical-related operations
 */
require_once __DIR__ . '/../../core/Auth.php';
require_once __DIR__ . '/../../core/Controller.php';
require_once __DIR__ . '/../../core/Database.php';
require_once __DIR__ . '/../../app/models/MedicalModel.php';
require_once __DIR__ . '/../../app/models/StudentModel.php';
require_once __DIR__ . '/../../app/helpers/SessionHelper.php';

class MedicalController extends Controller {
    private $model;
    private $studentModel;
    private $db;

    public function __construct() {
        Auth::checkRole(['Super Admin', 'Admin', 'Teacher', 'Medical Staff']);
        $this->model = new MedicalModel();
        $this->studentModel = new StudentModel();
        $this->db = Database::getInstance()->getConnection();
    }

    public function index() {
        $stats = [
            'total_visits' => $this->model->getTotalVisits(),
            'students_with_records' => $this->model->getTotalStudentsWithRecords(),
            'visits_this_month' => $this->model->getVisitsThisMonth(),
            'active_conditions' => $this->model->getActiveConditions(),
            'medication_value' => $this->model->getMedicationValue(),
            'low_stock' => count($this->model->getLowStockMedications())
        ];
        $recent_visits = $this->model->getAllVisits(null, null);
        $recent_visits = array_slice($recent_visits, 0, 10);
        require_once __DIR__ . '/../views/medical/dashboard.php';
    }

    public function students() {
        $students = $this->studentModel->getAll();
        require_once __DIR__ . '/../views/medical/students.php';
    }

    public function record($student_id) {
        $student = $this->studentModel->getById($student_id);
        $record = $this->model->getMedicalRecord($student_id);
        $conditions = $this->model->getStudentConditions($student_id);
        $visits = $this->model->getAllVisits($student_id);
        $immunizations = $this->model->getImmunizations($student_id);
        if (!$student) die("Student not found");
        require_once __DIR__ . '/../views/medical/record.php';
    }

    public function createRecord() {
        $students = $this->studentModel->getAll();
        $conditions = $this->model->getAllConditions();
        require_once __DIR__ . '/../views/medical/record_create.php';
    }

    public function storeRecord() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            
            // Check if record exists
            $existing = $this->model->getMedicalRecord($_POST['student_id']);
            
            if ($existing) {
                if ($this->model->updateMedicalRecord($_POST['student_id'], $_POST)) {
                    SessionHelper::setFlash('medical_msg', 'Medical record updated successfully!', 'success');
                }
            } else {
                if ($this->model->createMedicalRecord($_POST)) {
                    SessionHelper::setFlash('medical_msg', 'Medical record created successfully!', 'success');
                }
            }
            
            // Assign conditions
            if (isset($_POST['condition_ids']) && is_array($_POST['condition_ids'])) {
                foreach ($_POST['condition_ids'] as $condition_id) {
                    $this->model->assignCondition($_POST['student_id'], $condition_id, date('Y-m-d'));
                }
            }
            
            header("Location: /sms/public/medical/students");
            exit;
        }
    }

    public function visits() {
        $status = $_GET['status'] ?? null;
        $visits = $this->model->getAllVisits(null, $status);
        require_once __DIR__ . '/../views/medical/visits.php';
    }

    public function createVisit() {
        $students = $this->studentModel->getAll();
        $staff = $this->model->getAllStaff();
        require_once __DIR__ . '/../views/medical/visit_create.php';
    }

    public function storeVisit() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            $user = Auth::user();
            $_POST['attended_by'] = $user['id'];
            
            if ($this->model->createVisit($_POST)) {
                SessionHelper::setFlash('medical_msg', 'Medical visit recorded successfully!', 'success');
            }
            header("Location: /sms/public/medical/visits");
            exit;
        }
    }

    public function editVisit($id) {
        $visit = $this->model->getVisitById($id);
        $medications = $this->model->getAllMedications();
        if (!$visit) die("Visit not found");
        require_once __DIR__ . '/../views/medical/visit_edit.php';
    }

    public function updateVisit($id) {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            if ($this->model->updateVisit($id, $_POST)) {
                SessionHelper::setFlash('medical_msg', 'Visit updated successfully!', 'success');
            }
            header("Location: /sms/public/medical/visits");
            exit;
        }
    }

    public function closeVisit($id) {
        if ($this->model->closeVisit($id)) {
            SessionHelper::setFlash('medical_msg', 'Visit closed successfully!', 'success');
        }
        header("Location: /sms/public/medical/visits");
        exit;
    }

    public function dispense($visit_id) {
        $visit = $this->model->getVisitById($visit_id);
        $medications = $this->model->getAllMedications();
        if (!$visit) die("Visit not found");
        require_once __DIR__ . '/../views/medical/dispense.php';
    }

    public function storeDispense() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            $user = Auth::user();
            
            $_POST['dispensed_by'] = $user['id'];
            
            if ($this->model->dispenseMedication($_POST)) {
                // Update stock
                $this->model->updateStock($_POST['medication_id'], -$_POST['quantity']);
                
                // Log inventory
                $medication = $this->model->getMedicationById($_POST['medication_id']);
                $this->model->logInventory([
                    'medication_id' => $_POST['medication_id'],
                    'transaction_type' => 'Dispensed',
                    'quantity' => -$_POST['quantity'],
                    'balance_after' => $medication['stock_quantity'] - $_POST['quantity'],
                    'reference_number' => 'DISP-' . date('Ymd') . '-' . $_POST['visit_id'],
                    'notes' => 'Dispensed to student',
                    'created_by' => $user['id']
                ]);
                
                SessionHelper::setFlash('medical_msg', 'Medication dispensed successfully!', 'success');
            }
            header("Location: /sms/public/medical/visits");
            exit;
        }
    }

    public function medications() {
        $medications = $this->model->getAllMedications();
        $low_stock = $this->model->getLowStockMedications();
        $expired = $this->model->getExpiredMedications();
        require_once __DIR__ . '/../views/medical/medications.php';
    }

    public function createMedication() {
        require_once __DIR__ . '/../views/medical/medication_create.php';
    }

    public function storeMedication() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            $user = Auth::user();
            
            if ($this->model->createMedication($_POST)) {
                $med_id = $this->db->lastInsertId();
                // Log inventory
                $this->model->logInventory([
                    'medication_id' => $med_id,
                    'transaction_type' => 'Received',
                    'quantity' => $_POST['stock_quantity'],
                    'balance_after' => $_POST['stock_quantity'],
                    'reference_number' => 'REC-' . date('Ymd'),
                    'notes' => 'Initial stock',
                    'created_by' => $user['id']
                ]);
                
                SessionHelper::setFlash('medical_msg', 'Medication added successfully!', 'success');
            }
            header("Location: /sms/public/medical/medications");
            exit;
        }
    }

    public function editMedication($id) {
        $medication = $this->model->getMedicationById($id);
        if (!$medication) die("Medication not found");
        require_once __DIR__ . '/../views/medical/medication_edit.php';
    }

    public function updateMedication($id) {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            if ($this->model->updateMedication($id, $_POST)) {
                SessionHelper::setFlash('medical_msg', 'Medication updated successfully!', 'success');
            }
            header("Location: /sms/public/medical/medications");
            exit;
        }
    }

    public function stockAdjustment($id) {
        $medication = $this->model->getMedicationById($id);
        if (!$medication) die("Medication not found");
        require_once __DIR__ . '/../views/medical/stock_adjust.php';
    }

    public function storeStockAdjustment($id) {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            $user = Auth::user();
            
            $quantity_change = $_POST['quantity_change'];
            $medication = $this->model->getMedicationById($id);
            $new_balance = $medication['stock_quantity'] + $quantity_change;
            
            $this->model->updateStock($id, $quantity_change);
            
            $this->model->logInventory([
                'medication_id' => $id,
                'transaction_type' => 'Adjusted',
                'quantity' => $quantity_change,
                'balance_after' => $new_balance,
                'reference_number' => 'ADJ-' . date('Ymd') . '-' . $id,
                'notes' => $_POST['notes'],
                'created_by' => $user['id']
            ]);
            
            SessionHelper::setFlash('medical_msg', 'Stock adjusted successfully!', 'success');
            header("Location: /sms/public/medical/medications");
            exit;
        }
    }

    public function inventory() {
        $medication_id = $_GET['medication'] ?? null;
        $logs = $this->model->getInventoryLog($medication_id);
        $medications = $this->model->getAllMedications();
        require_once __DIR__ . '/../views/medical/inventory.php';
    }

    public function incidents() {
        $status = $_GET['status'] ?? null;
        $incidents = $this->model->getAllIncidents($status);
        require_once __DIR__ . '/../views/medical/incidents.php';
    }

    public function createIncident() {
        $students = $this->studentModel->getAll();
        require_once __DIR__ . '/../views/medical/incident_create.php';
    }

    public function storeIncident() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            $user = Auth::user();
            $_POST['reported_by'] = $user['id'];
            
            if ($this->model->createIncident($_POST)) {
                SessionHelper::setFlash('medical_msg', 'Incident reported successfully!', 'success');
            }
            header("Location: /sms/public/medical/incidents");
            exit;
        }
    }

    public function viewIncident($id) {
        $incident = $this->model->getIncidentById($id);
        if (!$incident) die("Incident not found");
        require_once __DIR__ . '/../views/medical/incident_view.php';
    }

    public function updateIncident($id) {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            if ($this->model->updateIncident($id, $_POST)) {
                SessionHelper::setFlash('medical_msg', 'Incident updated successfully!', 'success');
            }
            header("Location: /sms/public/medical/incidents");
            exit;
        }
    }

    public function immunizations() {
        $due = $this->model->getDueImmunizations();
        require_once __DIR__ . '/../views/medical/immunizations.php';
    }

    public function createImmunization() {
        $students = $this->studentModel->getAll();
        require_once __DIR__ . '/../views/medical/immunization_create.php';
    }

    public function storeImmunization() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            if ($this->model->createImmunization($_POST)) {
                SessionHelper::setFlash('medical_msg', 'Immunization recorded successfully!', 'success');
            }
            header("Location: /sms/public/medical/immunizations");
            exit;
        }
    }

    public function staff() {
        $staff = $this->model->getAllStaff();
        require_once __DIR__ . '/../views/medical/staff.php';
    }

    public function createStaff() {
        require_once __DIR__ . '/../views/medical/staff_create.php';
    }

    public function storeStaff() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            if ($this->model->createStaff($_POST)) {
                SessionHelper::setFlash('medical_msg', 'Staff added successfully!', 'success');
            }
            header("Location: /sms/public/medical/staff");
            exit;
        }
    }

    public function editStaff($id) {
        $staff = $this->model->getStaffById($id);
        if (!$staff) die("Staff not found");
        require_once __DIR__ . '/../views/medical/staff_edit.php';
    }

    public function updateStaff($id) {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            if ($this->model->updateStaff($id, $_POST)) {
                SessionHelper::setFlash('medical_msg', 'Staff updated successfully!', 'success');
            }
            header("Location: /sms/public/medical/staff");
            exit;
        }
    }

    public function deleteStaff($id) {
        if ($this->model->deleteStaff($id)) {
            SessionHelper::setFlash('medical_msg', 'Staff deleted successfully!', 'success');
        }
        header("Location: /sms/public/medical/staff");
        exit;
    }

    public function conditions() {
        $conditions = $this->model->getAllConditions();
        require_once __DIR__ . '/../views/medical/conditions.php';
    }

    public function createCondition() {
        require_once __DIR__ . '/../views/medical/condition_create.php';
    }

    public function storeCondition() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            if ($this->model->createCondition($_POST)) {
                SessionHelper::setFlash('medical_msg', 'Condition added successfully!', 'success');
            }
            header("Location: /sms/public/medical/conditions");
            exit;
        }
    }

    public function editCondition($id) {
        $condition = $this->model->getConditionById($id);
        if (!$condition) die("Condition not found");
        require_once __DIR__ . '/../views/medical/condition_edit.php';
    }

    public function updateCondition($id) {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            if ($this->model->updateCondition($id, $_POST)) {
                SessionHelper::setFlash('medical_msg', 'Condition updated successfully!', 'success');
            }
            header("Location: /sms/public/medical/conditions");
            exit;
        }
    }

    public function deleteCondition($id) {
        if ($this->model->deleteCondition($id)) {
            SessionHelper::setFlash('medical_msg', 'Condition deleted successfully!', 'success');
        }
        header("Location: /sms/public/medical/conditions");
        exit;
    }
}