<?php
require_once __DIR__ . '/../../core/Auth.php';
require_once __DIR__ . '/../../app/models/FeeModel.php';
require_once __DIR__ . '/../../app/models/StudentModel.php';
require_once __DIR__ . '/../../app/helpers/SessionHelper.php';

class FinanceController {
    private $model;
    private $studentModel;

    public function __construct() {
        // Only Accountants and Admins can access finance
        Auth::checkRole(['Super Admin', 'Admin', 'Accountant']);
        $this->model = new FeeModel();
        $this->studentModel = new StudentModel();
    }

    public function structure() {
        $feeTypes = $this->model->getFeeTypes();
        $classes = $this->studentModel->getClasses();
        require_once __DIR__ . '/../views/finance/structure.php';
    }

    public function saveStructure() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            $this->model->createStructure([
                'class_id' => $_POST['class_id'],
                'fee_type_id' => $_POST['fee_type_id'],
                'amount' => $_POST['amount'],
                'year' => $_POST['year']
            ]);
            SessionHelper::setFlash('fee_msg', 'Fee structure saved!', 'success');
            header("Location: /sms/public/finance/structure");
            exit;
        }
    }

    public function invoice() {
        $students = $this->studentModel->getAll(); // In prod, paginate this
        require_once __DIR__ . '/../views/finance/invoice_create.php';
    }

    public function generateInvoice() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            
            $student_id = $_POST['student_id'];
            $student = $this->studentModel->getById($student_id);
            $total = $_POST['total_amount'];
            
            $invoice_no = 'INV-' . date('Y') . '-' . rand(1000, 9999);

            $this->model->createInvoice([
                'invoice_no' => $invoice_no,
                'student_id' => $student_id,
                'class_id' => $student['class_id'],
                'total' => $total,
                'due_date' => $_POST['due_date']
            ]);

            SessionHelper::setFlash('fee_msg', 'Invoice generated!', 'success');
            header("Location: /sms/public/finance/invoices");
            exit;
        }
    }

    public function invoices() {
        // List all invoices (Simplified)
        $db = Database::getInstance()->getConnection();
        $invoices = $db->query("SELECT i.*, s.first_name, s.last_name FROM invoices i JOIN students s ON i.student_id = s.id ORDER BY i.created_at DESC")->fetchAll();
        require_once __DIR__ . '/../views/finance/invoices_list.php';
    }

    public function pay($id) {
        $invoice = $this->model->getInvoiceById($id);
        require_once __DIR__ . '/../views/finance/payment.php';
    }

    public function processPayment() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            $user = Auth::user();

            $invoice_id = $_POST['invoice_id'];
            $amount = $_POST['amount'];
            $invoice = $this->model->getInvoiceById($invoice_id);

            if ($amount > $invoice['balance_amount']) {
                SessionHelper::setFlash('fee_msg', 'Amount exceeds balance!', 'danger');
                header("Location: /sms/public/finance/pay/" . $invoice_id);
                exit;
            }

            // Record Payment
            $this->model->recordPayment([
                'invoice_id' => $invoice_id,
                'amount' => $amount,
                'date' => $_POST['payment_date'],
                'method' => $_POST['method'],
                'ref' => $_POST['ref'],
                'remarks' => $_POST['remarks'],
                'user_id' => $user['id']
            ]);

            // Update Invoice
            $new_paid = $invoice['paid_amount'] + $amount;
            $new_balance = $invoice['balance_amount'] - $amount;
            $status = ($new_balance == 0) ? 'Paid' : 'Partial';

            $this->model->updateInvoiceBalance($invoice_id, $amount, $new_balance, $status);

            SessionHelper::setFlash('fee_msg', 'Payment recorded!', 'success');
            header("Location: /sms/public/finance/receipt/" . $invoice_id);
            exit;
        }
    }

    public function receipt($id) {
        $invoice = $this->model->getInvoiceById($id);
        $payments = $this->model->getPaymentsByInvoice($id);
        require_once __DIR__ . '/../views/finance/receipt.php';
    }
}