<?php
/**
 * Dashboard Controller
 * Handles dashboard display and statistics
 */
require_once __DIR__ . '/../../core/Auth.php';
require_once __DIR__ . '/../../core/Database.php';
require_once __DIR__ . '/../../app/models/NotificationModel.php';
require_once __DIR__ . '/../../app/models/StudentModel.php';
require_once __DIR__ . '/../../app/models/FeeModel.php';

class DashboardController {
    private $notificationModel;
    private $studentModel;
    private $feeModel;
    private $db;

    public function __construct() {
        $this->notificationModel = new NotificationModel();
        $this->studentModel = new StudentModel();
        $this->feeModel = new FeeModel();
        $this->db = Database::getInstance()->getConnection();
    }

    public function index() {
        Auth::checkRole(['Super Admin', 'Admin', 'Teacher', 'Student', 'Parent', 'Accountant']);
        
        $user = Auth::user();
        $stats = $this->getStatistics($user['role_name']);
        $unreadNotifications = $this->notificationModel->getUnread($user['id']);
        $unreadCount = $this->notificationModel->getUnreadCount($user['id']);
        
        // Get recent activities based on role
        $recentActivities = $this->getRecentActivities($user['role_name']);
        
        require_once __DIR__ . '/../views/dashboard/index.php';
    }

    private function getStatistics($role) {
        $stats = [];
        
        if (in_array($role, ['Super Admin', 'Admin'])) {
            // Admin Statistics
            $stats['total_students'] = $this->db->query("SELECT COUNT(*) FROM students WHERE status = 1")->fetchColumn();
            $stats['total_teachers'] = $this->db->query("SELECT COUNT(*) FROM users WHERE role_id = 3 AND is_active = 1")->fetchColumn();
            $stats['total_parents'] = $this->db->query("SELECT COUNT(*) FROM parents_guardians WHERE status = 1")->fetchColumn();
            $stats['total_revenue'] = $this->db->query("SELECT SUM(paid_amount) FROM invoices")->fetchColumn() ?? 0;
            $stats['pending_fees'] = $this->db->query("SELECT SUM(balance_amount) FROM invoices WHERE balance_amount > 0")->fetchColumn() ?? 0;
            $stats['total_classes'] = $this->db->query("SELECT COUNT(*) FROM classes WHERE status = 1")->fetchColumn();
            $stats['total_books'] = $this->db->query("SELECT COUNT(*) FROM library_books WHERE status = 1")->fetchColumn();
            $stats['total_vehicles'] = $this->db->query("SELECT COUNT(*) FROM vehicles WHERE status = 1")->fetchColumn();
            $stats['boarding_students'] = $this->db->query("SELECT COUNT(*) FROM boarding_students WHERE status = 1")->fetchColumn();
            $stats['low_stock_medications'] = $this->db->query("SELECT COUNT(*) FROM medications WHERE stock_quantity <= reorder_level AND status = 1")->fetchColumn();
            
            // Attendance rate (last 30 days)
            $attendanceStats = $this->db->query("SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'Present' THEN 1 ELSE 0 END) as present
                FROM student_attendance 
                WHERE date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)")->fetch();
            $stats['attendance_rate'] = $attendanceStats['total'] > 0 
                ? round(($attendanceStats['present'] / $attendanceStats['total']) * 100, 2) 
                : 0;
            
            // Unpaid invoices count
            $stats['unpaid_invoices'] = $this->db->query("SELECT COUNT(*) FROM invoices WHERE status = 'Unpaid'")->fetchColumn();
            
        } elseif ($role == 'Teacher') {
            // Teacher Statistics
            $stats['total_students'] = $this->db->query("SELECT COUNT(*) FROM students WHERE status = 1")->fetchColumn();
            $stats['my_classes'] = $this->db->query("SELECT COUNT(DISTINCT class_id) FROM class_subjects WHERE teacher_id = " . $user['id'])->fetchColumn();
            $stats['pending_results'] = $this->db->query("SELECT COUNT(*) FROM exam_results WHERE entered_by = " . $user['id'])->fetchColumn();
            $stats['today_attendance'] = $this->db->query("SELECT COUNT(*) FROM student_attendance WHERE date = CURDATE()")->fetchColumn();
            $stats['total_revenue'] = 0;
            $stats['pending_fees'] = 0;
            $stats['attendance_rate'] = 0;
            
        } elseif ($role == 'Accountant') {
            // Accountant Statistics
            $stats['total_revenue'] = $this->db->query("SELECT SUM(paid_amount) FROM invoices")->fetchColumn() ?? 0;
            $stats['pending_fees'] = $this->db->query("SELECT SUM(balance_amount) FROM invoices WHERE balance_amount > 0")->fetchColumn() ?? 0;
            $stats['unpaid_invoices'] = $this->db->query("SELECT COUNT(*) FROM invoices WHERE status = 'Unpaid'")->fetchColumn();
            $stats['today_payments'] = $this->db->query("SELECT SUM(amount) FROM payments WHERE payment_date = CURDATE()")->fetchColumn() ?? 0;
            $stats['total_students'] = $this->db->query("SELECT COUNT(*) FROM students WHERE status = 1")->fetchColumn();
            $stats['total_teachers'] = 0;
            $stats['attendance_rate'] = 0;
            
        } elseif ($role == 'Student') {
            // Student Statistics
            $studentId = $this->getStudentIdByUserId($user['id']);
            $stats['attendance_rate'] = $studentId ? $this->getStudentAttendanceRate($studentId) : 0;
            $stats['total_revenue'] = 0;
            $stats['pending_fees'] = $studentId ? 
                ($this->db->prepare("SELECT SUM(balance_amount) FROM invoices WHERE student_id = ? AND balance_amount > 0")
                ->execute([$studentId]) ? $this->db->query("SELECT SUM(balance_amount) FROM invoices WHERE student_id = $studentId AND balance_amount > 0")->fetchColumn() ?? 0 : 0) 
                : 0;
            $stats['total_students'] = 0;
            $stats['total_teachers'] = 0;
            
        } elseif ($role == 'Parent') {
            // Parent Statistics
            $parentId = $this->db->prepare("SELECT id FROM parents_guardians WHERE email = ?")
                ->execute([$user['email']]) ? $this->db->query("SELECT id FROM parents_guardians WHERE email = '{$user['email']}'")->fetchColumn() : null;
            
            $stats['children_count'] = $parentId ? 
                $this->db->query("SELECT COUNT(*) FROM student_parents WHERE parent_id = $parentId")->fetchColumn() 
                : 0;
            $stats['pending_fees'] = $parentId ? 
                $this->db->query("SELECT SUM(i.balance_amount) FROM invoices i 
                JOIN student_parents sp ON i.student_id = sp.student_id 
                WHERE sp.parent_id = $parentId AND i.balance_amount > 0")->fetchColumn() ?? 0 
                : 0;
            $stats['total_revenue'] = 0;
            $stats['total_students'] = 0;
            $stats['total_teachers'] = 0;
            $stats['attendance_rate'] = 0;
            
        } else {
            // Default
            $stats = [
                'total_students' => 0,
                'total_teachers' => 0,
                'total_revenue' => 0,
                'pending_fees' => 0,
                'attendance_rate' => 0
            ];
        }
        
        return $stats;
    }

    private function getStudentIdByUserId($user_id) {
        // This would need a user-student relationship table
        // For now, return null
        return null;
    }

    private function getStudentAttendanceRate($student_id) {
        $stats = $this->db->prepare("SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'Present' THEN 1 ELSE 0 END) as present
            FROM student_attendance 
            WHERE student_id = ?")->execute([$student_id]) 
            ? $this->db->prepare("SELECT COUNT(*) as total, SUM(CASE WHEN status = 'Present' THEN 1 ELSE 0 END) as present FROM student_attendance WHERE student_id = ?")->execute([$student_id])->fetch() 
            : ['total' => 0, 'present' => 0];
        
        return $stats['total'] > 0 ? round(($stats['present'] / $stats['total']) * 100, 2) : 0;
    }

    private function getRecentActivities($role) {
        $activities = [];
        
        if (in_array($role, ['Super Admin', 'Admin'])) {
            // Recent payments
            $payments = $this->db->query("SELECT p.*, i.invoice_no, s.first_name, s.last_name 
                FROM payments p 
                JOIN invoices i ON p.invoice_id = i.id 
                JOIN students s ON i.student_id = s.id 
                ORDER BY p.created_at DESC LIMIT 5")->fetchAll();
            
            foreach ($payments as $p) {
                $activities[] = [
                    'type' => 'payment',
                    'message' => "Payment of $" . number_format($p['amount'], 2) . " received from " . $p['first_name'] . ' ' . $p['last_name'],
                    'date' => $p['created_at'],
                    'icon' => 'currency-dollar',
                    'color' => 'success'
                ];
            }
            
            // New students
            $students = $this->db->query("SELECT * FROM students ORDER BY created_at DESC LIMIT 3")->fetchAll();
            foreach ($students as $s) {
                $activities[] = [
                    'type' => 'student',
                    'message' => "New student admitted: " . $s['first_name'] . ' ' . $s['last_name'],
                    'date' => $s['created_at'],
                    'icon' => 'person-plus',
                    'color' => 'primary'
                ];
            }
            
        } elseif ($role == 'Teacher') {
            // Recent attendance
            $attendance = $this->db->query("SELECT sa.*, s.first_name, s.last_name 
                FROM student_attendance sa 
                JOIN students s ON sa.student_id = s.id 
                WHERE sa.date = CURDATE() 
                ORDER BY sa.created_at DESC LIMIT 5")->fetchAll();
            
            foreach ($attendance as $a) {
                $activities[] = [
                    'type' => 'attendance',
                    'message' => $a['first_name'] . ' ' . $a['last_name'] . " marked as " . $a['status'],
                    'date' => $a['created_at'],
                    'icon' => 'calendar-check',
                    'color' => 'info'
                ];
            }
        }
        
        return $activities;
    }
}